import re
import unicodedata
import logging
import pandas as pd
import numpy as np

# Función para normalizar textos
def normalizar_texto(s: str) -> str:
    s = str(s)
    s = unicodedata.normalize("NFD", s)
    s = "".join(ch for ch in s if unicodedata.category(ch) != "Mn")
    s = s.lower()
    s = re.sub(r"[^\w\s]", " ", s)
    return re.sub(r"\s+", " ", s).strip()

#Funcion para detectar posicion de las paginas de inicio y termino del documento entre las que se extraeran las tablas
def y_corte(page, pattern_norm, *, despues):
    """Coordenada Y del borde del patrón."""
    text_norm = normalizar_texto(page.extract_text() or "").replace(" ", "")
    idx = text_norm.find(pattern_norm)
    if idx == -1:
        return None
    char_sum = 0
    for w in page.extract_words():
        w_norm = normalizar_texto(w["text"]).replace(" ", "")
        prev = char_sum
        char_sum += len(w_norm)
        if despues and char_sum >= idx + len(pattern_norm):
            return w["bottom"]
        if not despues and prev <= idx < char_sum:
            return w["top"]
    return None

# Exclusión para documentos que tienen encabezados con formato tabla
def es_tabla_excepcion(tabla):
    """
    Excluye SOLO tablas de encabezado institucional de una columna
    """
    if not tabla or not tabla[0]:
        return False

    filas_validas = [r for r in tabla if any(str(c).strip() for c in r)]
    if not filas_validas:
        return False

    primera_fila = filas_validas[0]

    # SOLO excluir si:
    # 1. Es de una columna
    # 2. Y es exactamente un texto institucional
    if len(primera_fila) == 1:
        primera_celda_norm = normalizar_texto(str(primera_fila[0]))

        # Lista simplificada de textos a excluir (coincidencia exacta)
        textos_exactos = [
            "republica de chile",
            "comision de evaluacion",
            "region de la araucania"
        ]

        if primera_celda_norm in textos_exactos:
            return True

    return False  # En todos los demás casos, NO excluir

# Función para quitar saltos de lineas y espacios de valores que estan en varias líneas consecutivas
def limpiar_valor(txt: str) -> str:
    """
    Devuelve el texto sin saltos de línea ni espacios redundantes.
    • Sustituye \n y \r por un espacio.
    • Colapsa tabulaciones / múltiples espacios en un solo espacio.
    • Elimina espacios al inicio y al final.
    """
    if not txt:
        return ""
    txt = txt.replace("\n", " ").replace("\r", " ")
    txt = re.sub(r"\s+", " ", txt)
    return txt.strip()

BLACKLIST = {"para validar las firmas", "firmado digitalmente", "http", "rca n"
}

# Funcion para buscar encabezado cuando no existe o no se encuentra dentro de la tabla
def buscar_encabezado_simple(page, fila0, n_lineas=2):
    """
    Busca el texto inmediatamente superior a la tabla sin encabezado.
    Estrategia simplificada:
    1. Busca la etiqueta en la página actual, toma la línea anterior válida
    2. Si no encuentra, busca en la página anterior desde abajo hacia arriba
    3. Ignora líneas de la blacklist
    """
    etiqueta_norm = normalizar_texto(fila0[0])

    def es_linea_valida(linea):
        """Retorna True si la línea no está en blacklist y no está vacía"""
        if not linea.strip():
            return False
        linea_norm = normalizar_texto(linea)
        return not any(b in linea_norm for b in BLACKLIST)

    # ------------ 1) Página actual -----------------------------------------
    lines = (page.extract_text() or "").splitlines()
    for idx, ln in enumerate(lines):
        if etiqueta_norm in normalizar_texto(ln):
            logging.info(f"→ etiqueta de campo hallada en línea: {ln}")

            # Buscar n_lineas válidas inmediatamente anteriores
            encabezado = []
            for step in range(1, min(idx + 1, n_lineas + 3)):  # buscar máximo n_lineas+2 atrás
                if idx - step >= 0:
                    cand = lines[idx - step].strip()
                    if es_linea_valida(cand):
                        encabezado.insert(0, cand)  # insertar al inicio para mantener orden
                        if len(encabezado) >= n_lineas:  # ya tenemos suficientes líneas
                            break

            if encabezado:
                return "\n".join(encabezado)
            break

    # ------------ 2) Página anterior -------------------------------
    if page.page_number > 1:
        prev = page.pdf.pages[page.page_number - 2]
        prev_lines = (prev.extract_text() or "").splitlines()

        # Buscar desde abajo hacia arriba las primeras n_lineas válidas
        encabezado = []
        for ln in reversed(prev_lines):
            if es_linea_valida(ln):
                encabezado.insert(0, ln.strip())  # insertar al inicio para mantener orden
                if len(encabezado) >= n_lineas:   # ya tenemos suficientes líneas
                    break

        if encabezado:
            return "\n".join(encabezado)

    return None

# Funcion para unir textos de campos que se fusionan
def unir_textos(*valores):
    v = [str(x).strip() for x in valores if pd.notna(x) and str(x).strip()]
    return "\n".join(v) if v else np.nan
